/************************************************************************
 * NAME:	svd.c
 *
 * DESCR:	This is the high-level to the svd format.  You should
 *		also look at svd[0-9][0-9].[ch] which have specific
 *		code for the different versions of the SVD format.
 *		These were split out from this file to eliminate
 *		confusion and inadvertant changes to the old formats.
 ************************************************************************/
#include <sys/types.h>
#include <stdio.h>

#include "standard.h"
#include "floppy.h"
#include "error.h"
#include "svd.h"

/************************************************************************
 * NAME:	svd_report()
 *
 * DESCR:	Generates a report of the (apparently) svd file.
 *
 * ARGS:	level refers to the amount of reporting, 1 is lowest,
 *		3 is highest.
 *
 * RETURNS:	
 *
 * NOTES:	- levels other than 1 cause a detailed look at the file
 *		  beyond the guessing that occurred before
 ************************************************************************/
int
svd_report(int fd, struct floppy *floppy, int level)
{
    int	errors;
    int major, minor;

    lseek(fd,(off_t)0,SEEK_SET);

    errors = svd_read_version(fd,&major,&minor);

    printf("SVD %d.%d",major,minor);

    if (level == 1) {		/* first level, just indicate the type	*/
	printf("\n");
	return;
    }

    if (level == 2 || level == 3 || level == 4) {
	lseek(fd,(off_t)0,SEEK_SET);
	errors = svd_read(fd,floppy);
	floppy_report(floppy,level,errors);
    }
}


/************************************************************************
 * NAME:	svd_version_supported()
 *
 * DESCR:	Returns TRUE if the given version is supported.  False
 *		otherwise.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
svd_version_supported(int major, int minor)
{
    if (major == 1 && minor == 2) {
	return(TRUE);
    }
    if (major == 1 && minor == 5) {
	return(TRUE);
    }
    if (major == 2 && minor == 0) {
	return(TRUE);
    }

    return(FALSE);
}

/************************************************************************
 * NAME:	svd_read_line()
 *
 * DESCR:	Low level function to read a line...I know...I know...
 *		I'm reinventing stdio...
 *
 * ARGS:	
 *
 * RETURNS:	the number of characters read...includes the newline if
 *		there was one.  zero indicates end of file.
 *
 * NOTES:	- up to max characters will be read
 ************************************************************************/
int
svd_read_line(int fd, char *buffer, int max)
{
    int	i;

    for (i=0; i < max; buffer++) {
	if ( read(fd,buffer,1) == 0) {
	    break;
	}
	i++;
	if (*buffer == '\n') {
	    break;
	}
    }
    return(i);
}

/************************************************************************
 * NAME:	svd_read_version()
 *
 * DESCR:	Reads the version information from the given SVD file.
 *
 * ARGS:	
 *
 * RETURNS:	the version number is returned as a pair of ints.
 *
 * NOTES:	
 ************************************************************************/
int
svd_read_version(int fd, int *major, int *minor)
{
    char    linebuffer[80];
    char   *ptr;
    int	    count;

    count = svd_read_line(fd,linebuffer,sizeof(linebuffer));
    if (count == 0) {
	return(E_FATAL|E_SVD_BAD_FORMAT);
    }
    if (sscanf(linebuffer,"%d.%d",major,minor) != 2) {
	return(E_FATAL|E_SVD_BAD_FORMAT);
    }

    return(E_NONE);
}

/************************************************************************
 * NAME:	reflect()
 *
 * DESCR:	"reflects" bytes (MSb becomes LSb, and vice-versa).  Used
 *		by formats that send out LSb first.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
unsigned int
reflect(unsigned int in)
{
    unsigned int	out;

    out  = (in&0x01)<<7;
    out |= (in&0x02)<<5;
    out |= (in&0x04)<<3;
    out |= (in&0x08)<<1;
    out |= (in&0x10)>>1;
    out |= (in&0x20)>>3;
    out |= (in&0x40)>>5;
    out |= (in&0x80)>>7;

    return(out);
}

/************************************************************************
 * NAME:	svd_dump()
 *
 * DESCR:	Dumps svd in the most recent mode supported.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- change this if a more recent mode is added.
 ************************************************************************/
int
svd_dump(int fd,struct floppy *floppy)
{
    return(svd_dump_20(floppy,fd));
}

/************************************************************************
 * NAME:	svd_guesser()
 *
 * DESCR:	Looks at the incoming file and checks to see if it looks
 *		live an SVD file.  This is most obvious if there is the
 *		right set of stuff in the first sector, followed by the
 *		right zeros.  It may make sense later to look at a couple of
 *		tracks to make sure.  But that means that you have to 
 *		really look at the data in the header!
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if it looks like an svd.  FALSE otherwise.
 *
 * NOTES:	This may not be enough for decent REAL guessing...but I'm
 *		going with it for now.
 ************************************************************************/
int
svd_guesser(int fd)
{
    int		major, minor, sectors, tracks, secsize, sides, wprot;

    if (svd_read_version(fd,&major,&minor) != E_NONE) {
	return(FALSE);
    }

    if (!svd_version_supported(major,minor)) {
	return(FALSE);
    }

    if (major == 1 && minor == 2) {
	return(svd_read_12header(fd,&sectors,&tracks) == E_NONE);
    }
    if (major == 1 && minor == 5) {
	return(svd_read_15header(fd,&sectors,&tracks) == E_NONE);
    }
    if (major == 2 && minor == 0) {
	return(svd_read_20header(fd,&sectors,&tracks,&sides,&secsize,&wprot) == E_NONE);
    }

    /* other versions go here	*/

    return(FALSE);
}

/************************************************************************
 * NAME:	svd_read()
 *
 * DESCR:	Read in the svd file.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	This routine must be aware of different versions of the
 *		SVD file.
 ************************************************************************/
int
svd_read(int fd, struct floppy *floppy)
{
    FILE	*input;
    int		 major, minor;
    int		 r;

    if ((r = svd_read_version(fd, &major, &minor)) != E_NONE) {
	return(r);
    }

    if (!svd_version_supported(major,minor)) {
	return(E_FATAL|E_SVD_BAD_VERSION);
    }

    if (major == 1 && minor == 2) {
	return(svd_read_12(fd,floppy));
    }

    if (major == 1 && minor == 5) {
	return(svd_read_15(fd,floppy));
    }

    if (major == 2 && minor == 0) {
	return(svd_read_20(fd,floppy));
    }

    /* other versions go here	*/

    return(E_SVD_BAD_VERSION);
}


/************************************************************************
 * NAME:	svd_init()
 *
 * DESCR:	Registers the svd floppy format.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if the registration went OK, FALSE otherwise.
 *
 * NOTES:
 ************************************************************************/
int
svd_init()
{
    if(!floppy_format_register("SVD", "SVD Internal Format",svd_guesser, svd_read, svd_dump, svd_report)) {
	return(FALSE);
    }

    if (!floppy_fileext_register("svd", EXT_VERY_SPECIFIC, "SVD")) {
	return(FALSE);
    }

    if (!floppy_fileext_register("dsk", EXT_VERY_SPECIFIC, "SVD")) {
	return(FALSE);
    }

    return(TRUE);

}
